(* rewrite executable -- use FrontC library to rewrite a C file.
**
** Project: 		Calipso
** File: 			rewrite.ml
** Version:			2.0
** Date:			3.22.99
** Author:			Hugues Cass
**
*)

(* Useful Data *)
let version = "Rewrite V2.0 3.22.99 Hugues Cass"
let help = version ^ "\n"
	^ "rewrite [-V] [-P] [-t tab] [-w width] [-p preprocessor] <file list> [-o <output file>]"


(* Output management *)
let out = ref stdout
let close_me = ref false

let close_output _ =
	flush !out;
	if !close_me then close_out !out else ();
	close_me := false
let set_output filename =
	close_output ();
	(try out := open_out filename (*; prerr_string (filename ^ " opened\n.")*)
	with (Sys_error msg) ->
		output_string stderr ("Error while opening output: " ^ msg); exit 1);
	close_me := true
	

(* File Management *)
let files = ref []

let add_file filename =
	files := List.append !files [filename]


(*
** Argument definition
*)
let preproc = ref ""
let verbose = ref false
let arg_def =
[
	"-V", Arg.Unit (fun _ -> print_endline help), "Informations";
	"-v", Arg.Set verbose, "Verbose mode";
	"-o", Arg.String set_output, "Output file";
	"-t", Arg.Int Cprint.set_tab, "Identation step";
	"-w", Arg.Int Cprint.set_width, "Page width";
	"-p", Arg.String (fun id -> preproc := id), "Preprocessor command.";
	"-P", Arg.Unit (fun _ -> preproc := "gcc -E %i -o %o"), "Use \"gcc -E %i -o %o\" as preprocessor."
]


(* Starter *)
exception PreprocessingError
let preprocess inname outname =
	let rec replace str =
		try let idx = String.index str '%' in
			(if idx > 0 then String.sub str 0 idx else "")
			^ (match String.get str (idx + 1) with
				'i' -> inname
				| 'o' -> outname
				| '%' -> "%"
				| _ -> "")
			^ (if (idx + 2) >= (String.length str)
				then ""
				else replace
					(String.sub str (idx + 2) ((String.length str) - idx - 2))) 
		with Not_found -> str in
	let com = replace !preproc in
	let _ = if !verbose
		then prerr_string ("Executing \"" ^ com ^ "\"\n")
		else () in
	if (Sys.command com) = 0 
		then ()
		else raise PreprocessingError
	
let process filename =
		let parse file =
			let _ = if !verbose then prerr_string "Parsing...\n" else () in
			Frontc.parse_file file stderr in
		let _ = if !verbose
			then prerr_string ("Processing " ^ filename ^ "...\n")
			else () in
		match (
			if !preproc = ""
			then parse filename
			else
				let tmp = Filename.temp_file "rewrite" ".i" in
				try
					preprocess filename tmp;
					let res = parse tmp in
					Sys.remove tmp;
					res
				with PreprocessingError ->
					begin
						prerr_string
							("Error while preprocessing " ^ filename ^ "\n");
						Frontc.PARSING_ERROR
					end) with
		Frontc.PARSING_ERROR -> ()
		| Frontc.PARSING_OK defs ->
			output_string !out ("/* Generated by " ^ version ^ " */\n");
			if !verbose then prerr_string "Rewriting...\n";
			Cprint.print !out defs
	
let rec process_files files =
	List.iter process files

let _ =
	Arg.parse arg_def add_file help;
	process_files !files;
	close_output ()
