(* 
   This is a quotation expander for constructing C ASTs in the Cabs module
   which ships as part of the CDK. It's a little rough, and incomplete in 
   places, but is thurough enough for basic use in a code generator.
   
   Author: Graydon Hoare <graydon@redhat.com>
   Copyright (C) 2001, 2002 Red Hat.
   
   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
   
   You should have received a copy of the GNU Library General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA
   
*)


let c_lexer =  { Token.func = (Token.lexer_func_of_ocamllex Clex.c_token);
		 Token.using = (fun _ -> ());
		 Token.removing = (fun _ -> ());
		 Token.tparse = (fun _ -> None);
		 Token.text = Token.lexer_text }

let gram = Grammar.create (c_lexer)
;;

let cexpr = Grammar.Entry.create gram "cexpr"
let cstmt = Grammar.Entry.create gram "cstmt"
let cdecl = Grammar.Entry.create gram "cdecl"
let cfunc = Grammar.Entry.create gram "cfunc"
;;

let rec mk_list elts l =
  match elts with
      [] -> (let loc = l in <:expr< [] >>)
    | x::xs -> 
	( let loc = MLast.loc_of_expr x in
	    <:expr< [ $x$ :: $mk_list xs loc$ ] >> )
;;

let id_ctor e = e;;
let ptr_ctor loc ty = <:expr< Cabs.PTR $ty$ >>;;
let array_ctor loc dim ty = <:expr< Cabs.ARRAY $ty$ $dim$ >>;;
			    
let build_name l ty (name, ctor, init) =
  let loc = MLast.loc_of_expr name in
    <:expr< ($name$, $ctor ty$, [], $init$) >>
;;
  

let compose f g = (fun x -> f (g x))
 
let perhaps_expr loc option default =
  match option with
      None -> default
    | Some thing -> <:expr< $thing$ >>
;;

let perhaps option default =
  match option with 
      None -> default
    | Some thing -> thing
;;


let mk_stmts sz l =
  match (List.rev sz) with
      [] -> (let loc = l in <:expr< Cabs.NOP >>)
    | [s] -> s
    | v::vz ->
	List.fold_right 
	(fun s accum ->
	   let loc = MLast.loc_of_expr s in
	     <:expr< Cabs.SEQUENCE ($s$, $accum$) >>)
	(List.rev vz) v
;;

EXTEND
  GLOBAL: cexpr cstmt cdecl cfunc;


ident:
  [ [ x = IDENT -> 
	<:expr< $str:x$ >>

    | x = ANTI_IDENT -> 
	<:expr< $lid:x$ >>
    ] ]
;

(* expressions are the ugly bit *)

constant:
  [
    LEFTA
     [ x = ANTI_EXPR -> 
	 <:expr< $lid:x$ >>
	 
     | x = ANTI_STR ->
	 <:expr< Cabs.CONSTANT (Cabs.CONST_STRING $lid:x$) >>
	 
     | x = ANTI_FLO -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_FLOAT (string_of_float $lid:x$)) >>
	 
     | x = ANTI_CHAR -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_CHAR (Char.escaped $lid:x$)) >>
	 
     | x = ANTI_INT -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_INT (string_of_int $lid:x$)) >> 
	 
     | x = STRING -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_STRING $str:x$) >>
	 
     | x = FLOAT -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_FLOAT $str:x$) >>
	 
     | x = CHAR -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_CHAR $str:x$) >>
	 
     | x = INT -> 
	 <:expr< Cabs.CONSTANT (Cabs.CONST_INT $str:x$) >> ]
     
  | [ x = ident -> 
	<:expr< Cabs.VARIABLE $x$ >> ]
  ]
;


cexpr:
  [
    (* we need to separate out handling of COMMA operators, 
       since they clash with constant initializer syntax *)
    [e = LIST0 cexpr1 SEP "," -> 
       if List.length e = 1 
       then List.hd e
       else 
	 <:expr< Cabs.COMMA ($mk_list e loc$) >> ]
  ]
;

cexpr1:
  [ 
    (* prec level 15 *)
    [ "{"; es = LIST0 cexpr1 SEP ","; "}" -> 
	<:expr< Cabs.CONSTANT (Cabs.CONST_COMPOUND $mk_list es loc$) >> ]
    
  (* prec level 14 *)
  | LEFTA
      [ 
	fn = cexpr1; "("; args = LIST0 cexpr1 SEP ","; ")" -> 
	  <:expr< Cabs.CALL ($fn$, $mk_list args loc$) >> ]
      
  (* prec level 13 *)
  | RIGHTA   
      [ x = cexpr1; "+="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.ADD_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "-="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.SUB_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.ASSIGN, $x$, $y$) >>

      | x = cexpr1; "*="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.MUL_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "/="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.DIV_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "%="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.MOD_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "&="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.BAND_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "|="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.BOR_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "^="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.XOR_ASSIGN, $x$, $y$) >>

      | x = cexpr1; "<<="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.SHL_ASSIGN, $x$, $y$) >>
	  
      | x = cexpr1; ">>="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.SHR_ASSIGN, $x$, $y$) >> ]
      
  (* prec level 12 *)
  | RIGHTA
      [ q = cexpr1; "?"; res = cexpr1; ":"; alt = cexpr1 -> 
	  <:expr< Cabs.QUESTION ($q$, $res$, $alt$) >> ]
      
  (* prec level 11 *)
  | LEFTA
      [ x = cexpr1; "||"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.OR, $x$, $y$) >> ]

  (* prec level 10 *)
  | LEFTA
      [ x = cexpr1; "&&"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.AND, $x$, $y$) >> ]

  (* prec level 9 *)
  | LEFTA      
      [ x = cexpr1; "|"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.BOR, $x$, $y$) >> ]

  (* prec level 8 *)
  | LEFTA      
      [ x = cexpr1; "^"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.XOR, $x$, $y$) >> ]

  (* prec level 7 *)
  | LEFTA      
      [ x = cexpr1; "&"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.BAND, $x$, $y$) >> ]
      
  (* prec level 6 *)
  | LEFTA      
      [ x = cexpr1; "=="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.EQ, $x$, $y$) >>

      | x = cexpr1; "!="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.NE, $x$, $y$) >> ]
      
  (* prec level 5 *)
  | LEFTA      
      [ x = cexpr1; "<"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.LT, $x$, $y$) >>

      | x = cexpr1; "<="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.LE, $x$, $y$) >> 

      | x = cexpr1; ">"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.GT, $x$, $y$) >> 

      | x = cexpr1; ">="; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.GE, $x$, $y$) >> ]

  (* prec level 4 *)
      
  | LEFTA            
      [ x = cexpr1; "<<"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.SHL, $x$, $y$) >>
	  
      | x = cexpr1; ">>"; y = cexpr1 -> 
      <:expr< Cabs.BINARY (Cabs.SHR, $x$, $y$) >> ]
    

  (* prec level 3 *)
  | LEFTA      
      [ x = cexpr1; "+"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.ADD, $x$, $y$) >>

      | x = cexpr1; "-"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.SUB, $x$, $y$) >> ]

  (* prec level 2 *)
  | LEFTA      
      [ x = cexpr1; "*"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.MUL, $x$, $y$) >>

      | x = cexpr1; "/"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.DIV, $x$, $y$) >>

      | x = cexpr1; "%"; y = cexpr1 -> 
	  <:expr< Cabs.BINARY (Cabs.MOD, $x$, $y$) >> ]

  (* prec level 1 *)
  | RIGHTA
      [ "!"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.NOT, $y$) >>

      | "~"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.BNOT, $y$) >>

      | y = cexpr1; "++" -> 
	  <:expr< Cabs.UNARY (Cabs.POSINCR, $y$) >>

      | "++"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.PREINCR, $y$) >>

      | y = cexpr1; "--" -> 
	  <:expr< Cabs.UNARY (Cabs.POSDECR, $y$) >>

      | "--"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.PREDECR, $y$) >>

      | "+"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.PLUS, $y$) >>

      | "-"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.MINUS, $y$) >>

      | "*"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.MEMOF, $y$) >>

      | "&"; y = cexpr1 -> 
	  <:expr< Cabs.UNARY (Cabs.ADDROF, $y$) >> ]

  (* prec level 0 *)
  | LEFTA
      [ "("; e = cexpr; ")" -> 
	  <:expr< $e$ >> 

      (* a fib: casts should technically be in prec level 1 but we
	 need them in this level to get the correct left-factoring with
	 parenthesized expressions. perhaps someday camlp4 will factor
	 across precedence levels *)

      | "("; ty = base_type; ")"; e = cexpr1 -> 
	  <:expr< Cabs.CAST ($ty$, $e$) >>

      | x = cexpr1; "["; i = cexpr1; "]" -> 
	  <:expr< Cabs.INDEX ($x$,$i$) >> 

      | x = cexpr1; "."; mem = ident -> 
	  <:expr< Cabs.MEMBEROF ($x$, $mem$) >>

      | x = cexpr1; "->"; mem = ident -> 
	  <:expr< Cabs.MEMBEROFPTR ($x$, $mem$) >> ]

  (* prec level -1 *)
  | LEFTA
      [ x = constant -> x ]

  ]
;


(* type declarations are the _really_ ugly bit *)

sign_spec:
  [
    [ 
      "unsigned" -> <:expr< Cabs.UNSIGNED >> 
    | "signed" -> <:expr< Cabs.SIGNED >> 
    ]
  ]
;

sign:
  [ 
    [ x = OPT sign_spec -> 
	(match x with 
	     Some spec -> spec 
	   | None -> <:expr< Cabs.NO_SIGN >>) ] 
  ] 
;


enum_item:
  [ [ name = ident; "="; e = cexpr1 -> 
	<:expr< ($name$,$e$) >> 

    | name = ident -> 
	<:expr< ($name$, Cabs.NOTHING) >> ] ]
;

struct_member_name_ctor:
  [ [ p = ptype; name = ident; s = stype; i = init -> 
	(name, (compose s p), i) 
	
    | p = ptype; name = ident; ":"; e = cexpr1; i = init -> 
	(name, (fun _ -> <:expr< Cabs.BITFIELD (Cabs.NO_SIGN, $e$) >>), i) ] ]
;

struct_member_name_group: 
  [ [ sto = storage; ty = base_type; names = LIST1 struct_member_name_ctor SEP "," -> 
	let full_names = List.map (build_name loc ty) names in
	  <:expr< ($ty$, $sto$, $mk_list full_names loc$) >> ] ]
;
struct_or_union_member:
  [ [ ng = struct_member_name_group; ";" -> ng ] ]
;

base_type:
  [ 
    
    [ "void" -> 
	<:expr< Cabs.VOID >>

    | s = sign; "char" -> 
	<:expr< Cabs.CHAR $s$ >>

    | s = sign; "int" -> 
	<:expr< Cabs.INT (Cabs.NO_SIZE, $s$) >>

    | s = sign; "long"; OPT "int" -> 
	<:expr< Cabs.INT (Cabs.LONG, $s$) >>

    | s = sign; "long"; "long"; OPT "int" -> 
	<:expr< Cabs.INT (Cabs.LONG_LONG, $s$) >>

    | s = sign; "short"; OPT "int" -> 
	<:expr< Cabs.INT (Cabs.SHORT, $s$) >>

    | "enum"; name = OPT ident; "{"; items = LIST0 enum_item SEP ","; "}" ->
	let no_name = <:expr< "" >> in
	<:expr< Cabs.ENUM ($perhaps name no_name$, $mk_list items loc$) >>

    | "struct"; name = OPT ident; "{"; items = LIST0 struct_or_union_member; "}" -> 
	let no_name = <:expr< "" >> in
	<:expr< Cabs.STRUCT ($perhaps name no_name$, $mk_list items loc$) >>

    | "union"; name = OPT ident; "{"; items = LIST0 struct_or_union_member; "}" -> 
	let no_name = <:expr< "" >> in
	<:expr< Cabs.UNION ($perhaps name no_name$, $mk_list items loc$) >>

    | ty = base_type; "*" -> 
	<:expr< Cabs.PTR $ty$ >> 

    | "const"; ty = base_type -> 
	<:expr< Cabs.CONST $ty$ >> 

    | "volatile"; ty = base_type -> 
	<:expr< Cabs.VOLATILE $ty$ >> 

    | "struct"; name = ident -> 
	<:expr< Cabs.STRUCT ($name$, []) >>
	
    | "union"; name = ident -> 
	<:expr< Cabs.UNION ($name$, []) >> ]

  | [ x = ident -> 
	<:expr< Cabs.NAMED_TYPE $x$ >> ]

  ]
;
	
storage_spec: 
  [ [ "static" -> 
	<:expr< Cabs.STATIC >>

    | "auto" -> 
	<:expr< Cabs.AUTO >>

    | "extern" -> 
	<:expr< Cabs.EXTERN >>

    | "register" -> 
	<:expr< Cabs.REGISTER >> ] ] 
;

storage:  
  [ [ sto = OPT storage_spec -> 
	perhaps_expr loc sto <:expr< Cabs.NO_STORAGE >> ] ];

init_spec:  
  [ [ "="; v = cexpr1 -> v ] ] 
;
init: 
  [ [ v = OPT init_spec -> 
	perhaps_expr loc v <:expr< Cabs.NOTHING >> ] ] 
;

prefix_type_ctor_spec: 
  [ [ "*"; p = OPT prefix_type_ctor_spec -> 
	match p with 
	    None -> (ptr_ctor loc)
	  | Some ctor -> compose (ptr_ctor loc) ctor ] ]
;

ptype: 
  [ [ p = OPT prefix_type_ctor_spec -> 
	perhaps p id_ctor ] ]
; 

suffix_type_ctor_spec: 
  [ [ "["; e = OPT cexpr1; "]"; s = OPT suffix_type_ctor_spec -> 
	let dim = (perhaps e <:expr< Cabs.NOTHING >>) in
	  match s with
	      None -> (array_ctor loc dim)
	    | Some ctor -> compose (array_ctor loc dim) ctor ] ]
;

stype: 
  [ [ s = OPT suffix_type_ctor_spec -> 
	perhaps s id_ctor ] ]
;

name_ctor:
  [ [ p = ptype; name = ident; 
      s = stype; i = init -> (name, (compose s p), i) ] ]
;

name_group: 
  [ [ sto = storage; ty = base_type; names = LIST0 name_ctor SEP "," -> 
	let full_names = List.map (build_name loc ty) names in
	  <:expr< ($ty$, $sto$, $mk_list full_names loc$) >> ] ]
;

single_name_triple:
  [ [ sto = storage; ty = base_type; name = name_ctor ->
	  (ty, sto, name) ] ]
;

single_name: 
  [ [ triple = single_name_triple -> 
	let (ty,sto,name_ctor) = triple in
	let full_name = build_name loc ty name_ctor in
	  <:expr< ($ty$, $sto$, $full_name$) >> ] ]
;

cdecl:
  [ [ x = ANTI_DECL -> 
	<:expr< $lid:x$ >>

    | "typedef"; ng = name_group; ";" -> 
	<:expr< Cabs.TYPEDEF $ng$ >> 

    | ng = name_group; ";" -> 
	<:expr< Cabs.DECDEF $ng$ >> ] ]
;

body:
  [ [ x = ANTI_BODY -> <:expr< $lid:x$ >> ]
  | [ (* FIXME: this doesn't seem to factor well. not a killer, but
	 a little annoying.
	 decls = LIST0 cdecl; 
	 stmts = LIST0 cstmt1 -> 
	 <:expr< ($mk_list decls loc$, $mk_stmts stmts loc$) >>
	| *)
      stmts = cstmt -> <:expr< ([], $stmts$) >> 
    ] ]
;

cfunc:
  [ [ name = single_name_triple; 
      "("; args = LIST0 single_name SEP ","; ")";
      "{"; b = body; "}" -> 
	let (name_base_ty, storage, name_ctor) = name in
	let args_expr = mk_list args loc in
	let full_ty = 
	  <:expr< (Cabs.PROTO ($name_base_ty$, $args_expr$, False)) >> 
	in
	let full_name = build_name loc full_ty name_ctor in
	  <:expr< Cabs.FUNDEF (($full_ty$, $storage$, $full_name$), $b$) >>
    ] ]
;


(* statements are relatively tame *)

cstmt1:
  [ [ "if"; "("; test = cexpr; ")"; b = cstmt1; "else"; alt = cstmt1 -> 
	<:expr< Cabs.IF ($test$, $b$, $alt$) >>

    | "if"; "("; test = cexpr; ")"; b = cstmt1 -> 
	<:expr< Cabs.IF ($test$, $b$, Cabs.NOP) >>

    | "while"; "("; test = cexpr; ")"; b = cstmt1 -> 
	<:expr< Cabs.WHILE ($test$, $b$) >>

    | "do"; b = cstmt1; "while"; "("; test = cexpr; ")" -> 
	<:expr< Cabs.DOWHILE ($test$, $b$) >>

    | "for"; "("; beg = cexpr; ";"; 
      test = cexpr; ";"; inc = cexpr; ")"; 
      b = cstmt1 -> 
	<:expr< Cabs.FOR ($beg$, $test$, $inc$, $b$) >>

    | "break"; ";" -> 
	<:expr< Cabs.BREAK >>

    | "continue"; ";" -> 
	<:expr< Cabs.CONTINUE >>

    | "return"; v = cexpr; ";" -> 
	<:expr< Cabs.RETURN ($v$) >>

    | "switch"; "("; v = cexpr; ")"; arms = cstmt1 -> 
	<:expr< Cabs.SWITCH ($v$, $arms$) >>

    | "goto"; label = ident; ";" -> 
	<:expr< Cabs.GOTO ($label$) >>

    | "{"; b = body; "}" -> 
	<:expr< Cabs.BLOCK ($b$) >>

    | x = cexpr; ";" -> 
	<:expr< Cabs.COMPUTATION ($x$) >>

    | label = ident; ":"; b = cstmt -> 
	<:expr< Cabs.LABEL ($label$, $b$) >>

    | x = ANTI_STMT -> 
	<:expr< $lid:x$ >>

    ]
      
  | [ "case"; v = cexpr; ":"; arm = cstmt1 -> 
	  <:expr< Cabs.CASE ($v$, $arm$) >>
	  
    | "default"; ":"; b = cstmt -> 
	<:expr< Cabs.DEFAULT ($b$) >> ]
      
 ]
;

cstmt:
  [ 
    [ sz = LIST1 cstmt1 -> mk_stmts sz loc ]
  | [ x = ANTI_STMTS -> 
	<:expr< 
	match (List.length $lid:x$) with
	    [ 0 -> Cabs.NOP
	    | 1 -> List.hd $lid:x$
	    | _ -> List.fold_right
		(fun x accum -> Cabs.SEQUENCE (x, accum))
		(List.rev (List.tl (List.rev $lid:x$)))
		(List.hd (List.rev $lid:x$)) ]
	    >> ]
  ]
;


END
;;


(* now construct the quotation expanders *)

let add_quot_exast rule name =
  let exp s = Grammar.Entry.parse rule (Stream.of_string s) in
  let pat s = failwith "pat type not implemented" in
    Quotation.add name (Quotation.ExAst (exp, pat))

let _ = List.iter
	  (fun (x, y) -> add_quot_exast x y)
	  [
	    (cstmt, "cstmt");
	    (cexpr, "cexpr");
	    (cdecl, "cdecl");
	    (cfunc, "cfunc")
	  ]
	  
let _ = Quotation.default := "cexpr"

